'use client';

import { useEffect, useState } from 'react';
import { productApi } from '@/lib/api';

export default function CategoriesPage() {
  const [categories, setCategories] = useState<any[]>([]);
  const [loading, setLoading] = useState(true);
  const [editing, setEditing] = useState<string | null>(null);
  const [newName, setNewName] = useState('');
  const [showNew, setShowNew] = useState(false);
  const [createName, setCreateName] = useState('');

  useEffect(() => {
    productApi.listCategories().then(setCategories).finally(() => setLoading(false));
  }, []);

  const handleUpdate = async (id: string) => {
    if (!newName.trim()) return;
    try {
      await productApi.updateCategory(id, { name: newName });
      setCategories((c) => c.map((x) => (x.id === id ? { ...x, name: newName } : x)));
      setEditing(null);
    } catch (e) {
      console.error(e);
    }
  };

  const handleCreate = async (e: React.FormEvent) => {
    e.preventDefault();
    if (!createName.trim()) return;
    try {
      const cat = await productApi.createCategory({ name: createName });
      setCategories((c) => [...c, cat]);
      setCreateName('');
      setShowNew(false);
    } catch (err) {
      console.error(err);
    }
  };

  const handleDelete = async (id: string) => {
    if (!confirm('Bu kategoriyi silmek istediğinize emin misiniz? Ürünler başka kategoriye taşınmalı.')) return;
    try {
      await productApi.deleteCategory(id);
      setCategories((c) => c.filter((x) => x.id !== id));
    } catch (err: any) {
      alert(err.message || 'Silinemedi');
    }
  };

  if (loading) return <div className="text-slate-500">Yükleniyor...</div>;

  return (
    <div>
      <div className="flex justify-between items-center mb-6">
        <h1 className="text-2xl font-bold text-slate-800">Kategoriler</h1>
        {!showNew && (
          <button
            onClick={() => setShowNew(true)}
            className="px-4 py-2 bg-emerald-600 text-white rounded-lg text-sm font-medium hover:bg-emerald-700"
          >
            Yeni kategori
          </button>
        )}
      </div>
      {showNew && (
        <form onSubmit={handleCreate} className="mb-4 flex gap-2">
          <input
            type="text"
            value={createName}
            onChange={(e) => setCreateName(e.target.value)}
            placeholder="Kategori adı"
            className="flex-1 px-4 py-2 border border-slate-300 rounded-lg"
            autoFocus
          />
          <button type="submit" className="px-4 py-2 bg-emerald-600 text-white rounded-lg">
            Ekle
          </button>
          <button type="button" onClick={() => { setShowNew(false); setCreateName(''); }} className="px-4 py-2 border rounded-lg">
            İptal
          </button>
        </form>
      )}
      <div className="bg-white rounded-xl border border-slate-200 overflow-hidden">
        <table className="w-full">
          <thead className="bg-slate-50 border-b border-slate-200">
            <tr>
              <th className="text-left py-3 px-4 text-sm font-medium text-slate-600">Ad</th>
              <th className="text-left py-3 px-4 text-sm font-medium text-slate-600">Ürün sayısı</th>
              <th className="w-40" />
            </tr>
          </thead>
          <tbody>
            {categories.map((c) => (
              <tr key={c.id} className="border-b border-slate-100 hover:bg-slate-50/50">
                <td className="py-3 px-4">
                  {editing === c.id ? (
                    <input
                      type="text"
                      value={newName}
                      onChange={(e) => setNewName(e.target.value)}
                      onBlur={() => handleUpdate(c.id)}
                      onKeyDown={(e) => e.key === 'Enter' && handleUpdate(c.id)}
                      className="w-full px-2 py-1 border rounded"
                      autoFocus
                    />
                  ) : (
                    <span
                      className="font-medium text-slate-800 cursor-pointer"
                      onClick={() => { setEditing(c.id); setNewName(c.name); }}
                    >
                      {c.name}
                    </span>
                  )}
                </td>
                <td className="py-3 px-4 text-slate-600">{c._count?.products ?? 0}</td>
                <td className="py-3 px-4">
                  <button
                    type="button"
                    onClick={() => handleDelete(c.id)}
                    className="text-red-600 hover:underline text-sm"
                  >
                    Sil
                  </button>
                </td>
              </tr>
            ))}
          </tbody>
        </table>
      </div>
    </div>
  );
}
