import { Injectable, NotFoundException, ForbiddenException } from '@nestjs/common';
import { PrismaService } from '../prisma/prisma.service';
import { AuditService } from '../audit/audit.service';
import type { CreateLocationBody, UpdateLocationBody, CreateTableBody, UpdateTableBody } from '@3dqr/shared';

@Injectable()
export class TenantService {
  constructor(
    private prisma: PrismaService,
    private audit: AuditService,
  ) {}

  async getTenantForUser(userId: string, tenantId: string) {
    const t = await this.prisma.tenant.findFirst({
      where: { id: tenantId, deletedAt: null, users: { some: { userId } } },
      include: {
        subscription: true,
        _count: { select: { locations: true, products: true, categories: true } },
      },
    });
    if (!t) throw new NotFoundException('Tenant bulunamadı');
    return t;
  }

  async getTenantBySlug(slug: string) {
    const t = await this.prisma.tenant.findFirst({
      where: { slug, deletedAt: null },
      include: { subscription: true },
    });
    if (!t) throw new NotFoundException('Menü bulunamadı');
    if (t.subscription?.status !== 'active' && t.subscription?.status !== 'trialing') {
      throw new ForbiddenException('Bu menü şu an kullanılamıyor');
    }
    return t;
  }

  async listLocations(tenantId: string) {
    return this.prisma.location.findMany({
      where: { tenantId },
      orderBy: { name: 'asc' },
      include: { _count: { select: { tables: true } } },
    });
  }

  async createLocation(tenantId: string, userId: string, body: CreateLocationBody) {
    const loc = await this.prisma.location.create({
      data: { tenantId, ...body },
    });
    await this.audit.log({
      userId,
      tenantId,
      action: 'create',
      resource: 'location',
      resourceId: loc.id,
      metadata: { name: loc.name },
    });
    return loc;
  }

  async updateLocation(tenantId: string, userId: string, locationId: string, body: UpdateLocationBody) {
    await this.ensureLocationBelongsToTenant(tenantId, locationId);
    const loc = await this.prisma.location.update({
      where: { id: locationId },
      data: body,
    });
    await this.audit.log({
      userId,
      tenantId,
      action: 'update',
      resource: 'location',
      resourceId: locationId,
    });
    return loc;
  }

  async deleteLocation(tenantId: string, userId: string, locationId: string) {
    await this.ensureLocationBelongsToTenant(tenantId, locationId);
    await this.prisma.location.delete({ where: { id: locationId } });
    await this.audit.log({
      userId,
      tenantId,
      action: 'delete',
      resource: 'location',
      resourceId: locationId,
    });
    return { ok: true };
  }

  async listTables(tenantId: string, locationId?: string) {
    const where: { location: { tenantId: string }; locationId?: string } = {
      location: { tenantId },
    };
    if (locationId) where.locationId = locationId;
    return this.prisma.table.findMany({
      where,
      orderBy: { name: 'asc' },
      include: { location: { select: { id: true, name: true } } },
    });
  }

  async createTable(tenantId: string, userId: string, body: CreateTableBody) {
    await this.ensureLocationBelongsToTenant(tenantId, body.locationId);
    const table = await this.prisma.table.create({
      data: body,
      include: { location: true },
    });
    await this.audit.log({
      userId,
      tenantId,
      action: 'create',
      resource: 'table',
      resourceId: table.id,
      metadata: { name: table.name },
    });
    return table;
  }

  async updateTable(tenantId: string, userId: string, tableId: string, body: UpdateTableBody) {
    await this.ensureTableBelongsToTenant(tenantId, tableId);
    if (body.locationId) await this.ensureLocationBelongsToTenant(tenantId, body.locationId);
    const table = await this.prisma.table.update({
      where: { id: tableId },
      data: body,
      include: { location: true },
    });
    await this.audit.log({
      userId,
      tenantId,
      action: 'update',
      resource: 'table',
      resourceId: tableId,
    });
    return table;
  }

  async deleteTable(tenantId: string, userId: string, tableId: string) {
    await this.ensureTableBelongsToTenant(tenantId, tableId);
    await this.prisma.table.delete({ where: { id: tableId } });
    await this.audit.log({
      userId,
      tenantId,
      action: 'delete',
      resource: 'table',
      resourceId: tableId,
    });
    return { ok: true };
  }

  async getPublicMenu(tenantSlug: string, locationId?: string, tableId?: string) {
    const tenant = await this.getTenantBySlug(tenantSlug);
    const categories = await this.prisma.category.findMany({
      where: { tenantId: tenant.id },
      orderBy: { sortOrder: 'asc' },
      include: {
        products: {
          where: { isActive: true },
          orderBy: { name: 'asc' },
        },
      },
    });
    let location = null;
    let table = null;
    if (locationId) {
      location = await this.prisma.location.findFirst({
        where: { id: locationId, tenantId: tenant.id },
      });
    }
    if (tableId && location) {
      table = await this.prisma.table.findFirst({
        where: { id: tableId, locationId: location.id },
      });
    }
    return {
      tenant: {
        id: tenant.id,
        name: tenant.name,
        slug: tenant.slug,
      },
      location: location ? { id: location.id, name: location.name } : null,
      table: table ? { id: table.id, name: table.name } : null,
      categories: categories.map((c) => ({
        ...c,
        products: c.products.map((p) => ({
          ...p,
          price: Number(p.price),
          allergens: (p.allergens as string[]) ?? [],
        })),
      })),
    };
  }

  private async ensureLocationBelongsToTenant(tenantId: string, locationId: string) {
    const loc = await this.prisma.location.findFirst({
      where: { id: locationId, tenantId },
    });
    if (!loc) throw new NotFoundException('Şube bulunamadı');
  }

  private async ensureTableBelongsToTenant(tenantId: string, tableId: string) {
    const t = await this.prisma.table.findFirst({
      where: { id: tableId, location: { tenantId } },
    });
    if (!t) throw new NotFoundException('Masa bulunamadı');
  }
}
