import { Injectable, NotFoundException, BadRequestException } from '@nestjs/common';
import { PrismaService } from '../prisma/prisma.service';
import { AuditService } from '../audit/audit.service';
import type {
  CreateCategoryBody,
  UpdateCategoryBody,
  CreateProductBody,
  UpdateProductBody,
  BulkProductActionBody,
} from '@3dqr/shared';

@Injectable()
export class ProductService {
  constructor(
    private prisma: PrismaService,
    private audit: AuditService,
  ) {}

  async listCategories(tenantId: string) {
    return this.prisma.category.findMany({
      where: { tenantId },
      orderBy: { sortOrder: 'asc' },
      include: { _count: { select: { products: true } } },
    });
  }

  async createCategory(tenantId: string, userId: string, body: CreateCategoryBody) {
    const cat = await this.prisma.category.create({
      data: { tenantId, ...body },
    });
    await this.audit.log({
      userId,
      tenantId,
      action: 'create',
      resource: 'category',
      resourceId: cat.id,
      metadata: { name: cat.name },
    });
    return cat;
  }

  async updateCategory(tenantId: string, userId: string, categoryId: string, body: UpdateCategoryBody) {
    await this.ensureCategoryBelongsToTenant(tenantId, categoryId);
    const cat = await this.prisma.category.update({
      where: { id: categoryId },
      data: body,
    });
    await this.audit.log({
      userId,
      tenantId,
      action: 'update',
      resource: 'category',
      resourceId: categoryId,
    });
    return cat;
  }

  async deleteCategory(tenantId: string, userId: string, categoryId: string) {
    await this.ensureCategoryBelongsToTenant(tenantId, categoryId);
    const count = await this.prisma.product.count({ where: { categoryId } });
    if (count > 0) throw new BadRequestException('Bu kategoride ürün var. Önce ürünleri taşıyın veya silin.');
    await this.prisma.category.delete({ where: { id: categoryId } });
    await this.audit.log({
      userId,
      tenantId,
      action: 'delete',
      resource: 'category',
      resourceId: categoryId,
    });
    return { ok: true };
  }

  async listProducts(tenantId: string, categoryId?: string, activeOnly?: boolean) {
    const where: { tenantId: string; categoryId?: string; isActive?: boolean } = { tenantId };
    if (categoryId) where.categoryId = categoryId;
    if (activeOnly === true) where.isActive = true;
    return this.prisma.product.findMany({
      where,
      orderBy: [{ category: { sortOrder: 'asc' } }, { name: 'asc' }],
      include: { category: { select: { id: true, name: true, sortOrder: true } } },
    });
  }

  async getProduct(tenantId: string, productId: string) {
    await this.ensureProductBelongsToTenant(tenantId, productId);
    const p = await this.prisma.product.findUnique({
      where: { id: productId },
      include: { category: true },
    });
    if (!p) throw new NotFoundException('Ürün bulunamadı');
    return { ...p, price: Number(p.price), allergens: (p.allergens as string[]) ?? [] };
  }

  async createProduct(tenantId: string, userId: string, body: CreateProductBody) {
    await this.ensureCategoryBelongsToTenant(tenantId, body.categoryId);
    const product = await this.prisma.product.create({
      data: {
        tenantId,
        categoryId: body.categoryId,
        name: body.name,
        description: body.description,
        price: body.price,
        currency: body.currency ?? 'TRY',
        allergens: body.allergens ?? undefined,
        isActive: body.isActive ?? true,
        imageUrls: body.imageUrls ?? [],
        model3dUrl: body.model3dUrl ?? undefined,
      },
      include: { category: true },
    });
    await this.audit.log({
      userId,
      tenantId,
      action: 'create',
      resource: 'product',
      resourceId: product.id,
      metadata: { name: product.name },
    });
    return { ...product, price: Number(product.price) };
  }

  async updateProduct(tenantId: string, userId: string, productId: string, body: UpdateProductBody) {
    await this.ensureProductBelongsToTenant(tenantId, productId);
    if (body.categoryId) await this.ensureCategoryBelongsToTenant(tenantId, body.categoryId);
    const product = await this.prisma.product.update({
      where: { id: productId },
      data: {
        ...(body.name !== undefined && { name: body.name }),
        ...(body.description !== undefined && { description: body.description }),
        ...(body.price !== undefined && { price: body.price }),
        ...(body.currency !== undefined && { currency: body.currency }),
        ...(body.allergens !== undefined && { allergens: body.allergens }),
        ...(body.categoryId !== undefined && { categoryId: body.categoryId }),
        ...(body.isActive !== undefined && { isActive: body.isActive }),
        ...(body.imageUrls !== undefined && { imageUrls: body.imageUrls }),
        ...(body.model3dUrl !== undefined && { model3dUrl: body.model3dUrl }),
      },
      include: { category: true },
    });
    await this.audit.log({
      userId,
      tenantId,
      action: 'update',
      resource: 'product',
      resourceId: productId,
    });
    return { ...product, price: Number(product.price) };
  }

  async deleteProduct(tenantId: string, userId: string, productId: string) {
    await this.ensureProductBelongsToTenant(tenantId, productId);
    await this.prisma.product.delete({ where: { id: productId } });
    await this.audit.log({
      userId,
      tenantId,
      action: 'delete',
      resource: 'product',
      resourceId: productId,
    });
    return { ok: true };
  }

  async bulkAction(tenantId: string, userId: string, body: BulkProductActionBody) {
    for (const productId of body.productIds) {
      await this.ensureProductBelongsToTenant(tenantId, productId);
    }
    if (body.action === 'move_category' && body.categoryId) {
      await this.ensureCategoryBelongsToTenant(tenantId, body.categoryId);
      await this.prisma.product.updateMany({
        where: { id: { in: body.productIds } },
        data: { categoryId: body.categoryId },
      });
    } else if (body.action === 'set_active') {
      await this.prisma.product.updateMany({
        where: { id: { in: body.productIds } },
        data: { isActive: true },
      });
    } else if (body.action === 'set_inactive') {
      await this.prisma.product.updateMany({
        where: { id: { in: body.productIds } },
        data: { isActive: false },
      });
    }
    await this.audit.log({
      userId,
      tenantId,
      action: 'bulk',
      resource: 'product',
      metadata: { action: body.action, count: body.productIds.length, categoryId: body.categoryId },
    });
    return { updated: body.productIds.length };
  }

  private async ensureCategoryBelongsToTenant(tenantId: string, categoryId: string) {
    const c = await this.prisma.category.findFirst({
      where: { id: categoryId, tenantId },
    });
    if (!c) throw new NotFoundException('Kategori bulunamadı');
  }

  private async ensureProductBelongsToTenant(tenantId: string, productId: string) {
    const p = await this.prisma.product.findFirst({
      where: { id: productId, tenantId },
    });
    if (!p) throw new NotFoundException('Ürün bulunamadı');
  }
}
