import {
  Controller,
  Get,
  Post,
  Put,
  Delete,
  Body,
  Param,
  Query,
  UseGuards,
  Req,
} from '@nestjs/common';
import { ProductService } from './product.service';
import { JwtAuthGuard, TenantGuard, SubscriptionGuard, RolesGuard, Roles } from '../auth/guards';
import { CreateCategoryDto, UpdateCategoryDto } from './dto/category.dto';
import { CreateProductDto, UpdateProductDto } from './dto/product.dto';
import { BulkProductActionDto } from './dto/bulk.dto';

@Controller('products')
@UseGuards(JwtAuthGuard, TenantGuard, SubscriptionGuard)
export class ProductController {
  constructor(private product: ProductService) {}

  @Get('categories')
  async listCategories(@Req() req: any) {
    return this.product.listCategories(req.tenantId);
  }

  @Post('categories')
  @Roles('owner', 'admin', 'editor')
  async createCategory(@Req() req: any, @Body() body: CreateCategoryDto) {
    return this.product.createCategory(req.tenantId, req.user.id, body);
  }

  @Put('categories/:id')
  @Roles('owner', 'admin', 'editor')
  async updateCategory(
    @Req() req: any,
    @Param('id') id: string,
    @Body() body: UpdateCategoryDto,
  ) {
    return this.product.updateCategory(req.tenantId, req.user.id, id, body);
  }

  @Delete('categories/:id')
  @Roles('owner', 'admin')
  async deleteCategory(@Req() req: any, @Param('id') id: string) {
    return this.product.deleteCategory(req.tenantId, req.user.id, id);
  }

  @Get()
  async listProducts(@Req() req: any, @Query('categoryId') categoryId?: string, @Query('activeOnly') activeOnly?: string) {
    return this.product.listProducts(
      req.tenantId,
      categoryId,
      activeOnly === 'true',
    );
  }

  @Get(':id')
  async getProduct(@Req() req: any, @Param('id') id: string) {
    return this.product.getProduct(req.tenantId, id);
  }

  @Post()
  @Roles('owner', 'admin', 'editor')
  async createProduct(@Req() req: any, @Body() body: CreateProductDto) {
    return this.product.createProduct(req.tenantId, req.user.id, body);
  }

  @Put(':id')
  @Roles('owner', 'admin', 'editor')
  async updateProduct(
    @Req() req: any,
    @Param('id') id: string,
    @Body() body: UpdateProductDto,
  ) {
    return this.product.updateProduct(req.tenantId, req.user.id, id, body);
  }

  @Delete(':id')
  @Roles('owner', 'admin')
  async deleteProduct(@Req() req: any, @Param('id') id: string) {
    return this.product.deleteProduct(req.tenantId, req.user.id, id);
  }

  @Post('bulk')
  @Roles('owner', 'admin', 'editor')
  async bulkAction(@Req() req: any, @Body() body: BulkProductActionDto) {
    return this.product.bulkAction(req.tenantId, req.user.id, body);
  }
}
