import { PrismaClient, UserRole, SubscriptionStatus } from '@prisma/client';
import * as bcrypt from 'bcrypt';

const prisma = new PrismaClient();

async function main() {
  const passwordHash = await bcrypt.hash('Admin123!', 10);

  // Super Admin
  const superAdmin = await prisma.user.upsert({
    where: { email: 'admin@3dqr.com' },
    update: {},
    create: {
      email: 'admin@3dqr.com',
      passwordHash,
      name: 'Super Admin',
      platformRole: 'super_admin',
    },
  });
  console.log('Super Admin:', superAdmin.email);

  // Example tenant owner
  const tenantOwner = await prisma.user.upsert({
    where: { email: 'owner@demo-restaurant.com' },
    update: {},
    create: {
      email: 'owner@demo-restaurant.com',
      passwordHash,
      name: 'Demo Restoran Sahibi',
    },
  });

  const tenant = await prisma.tenant.upsert({
    where: { slug: 'demo-restaurant' },
    update: {},
    create: {
      name: 'Demo Restoran',
      slug: 'demo-restaurant',
    },
  });

  await prisma.userTenant.upsert({
    where: {
      userId_tenantId: { userId: tenantOwner.id, tenantId: tenant.id },
    },
    update: {},
    create: {
      userId: tenantOwner.id,
      tenantId: tenant.id,
      role: UserRole.owner,
    },
  });

  await prisma.subscription.upsert({
    where: { tenantId: tenant.id },
    update: {},
    create: {
      tenantId: tenant.id,
      status: SubscriptionStatus.active,
      currentPeriodEnd: new Date(Date.now() + 30 * 24 * 60 * 60 * 1000),
    },
  });

  const location = await prisma.location.create({
    data: {
      tenantId: tenant.id,
      name: 'Ana Şube',
      address: 'Örnek Mah. Menü Sok. No:1',
      city: 'İstanbul',
      country: 'Türkiye',
    },
  });

  await prisma.table.createMany({
    data: [
      { locationId: location.id, name: 'Masa 1' },
      { locationId: location.id, name: 'Masa 2' },
      { locationId: location.id, name: 'Masa 3' },
    ],
  });

  const category1 = await prisma.category.create({
    data: { tenantId: tenant.id, name: 'İçecekler', sortOrder: 0 },
  });
  const category2 = await prisma.category.create({
    data: { tenantId: tenant.id, name: 'Ana Yemekler', sortOrder: 1 },
  });

  await prisma.product.createMany({
    data: [
      {
        tenantId: tenant.id,
        categoryId: category1.id,
        name: 'Türk Kahvesi',
        description: 'Geleneksel Türk kahvesi',
        price: 45,
        currency: 'TRY',
        isActive: true,
        imageUrls: [],
      },
      {
        tenantId: tenant.id,
        categoryId: category1.id,
        name: 'Ayran',
        description: 'Soğuk ayran',
        price: 25,
        currency: 'TRY',
        isActive: true,
        imageUrls: [],
      },
      {
        tenantId: tenant.id,
        categoryId: category2.id,
        name: 'Köfte',
        description: 'El yapımı köfte, pilav ve salata',
        price: 185,
        currency: 'TRY',
        allergens: ['Gluten'],
        isActive: true,
        imageUrls: [],
      },
    ],
  });

  console.log('Seed completed. Demo tenant slug: demo-restaurant');
}

main()
  .catch((e) => {
    console.error(e);
    process.exit(1);
  })
  .finally(() => prisma.$disconnect());
